#!/usr/bin/env python3
"""
Check for malformed links in the database.
"""

import sys
from db_utils import MigrationDB

def check_malformed_links():
    """Check for malformed links in the database."""
    db = MigrationDB()

    print("🔍 Checking for malformed links in database...")
    print("=" * 50)

    with db.get_connection() as conn:
        # Search for links containing the malformed patterns
        patterns = [
            '%$baseurl%',
            '%<?php%',
            '%intranet/databases/crm%'
        ]

        for pattern in patterns:
            links = conn.execute("SELECT id, source_file_id, target_path FROM links WHERE target_path LIKE ?", (pattern,)).fetchall()

            if links:
                print(f"Found {len(links)} links matching pattern '{pattern}':")
                for link in links[:10]:  # Show first 10
                    print(f"  ID {link['id']}: {link['target_path']} (from file {link['source_file_id']})")
                if len(links) > 10:
                    print(f"  ... and {len(links) - 10} more")
            else:
                print(f"No links found matching pattern '{pattern}'")

        # Specifically check for the exact malformed paths
        exact_malformed = [
            '/$baseurl/intranet/databases/crm/HiddenRemovedContacts.php',
            '/<?php print "$baseurl";?>/intranet/databases/crm/crmlinkeddocuments.php'
        ]

        print("\nChecking for exact malformed paths:")
        for path in exact_malformed:
            links = conn.execute("SELECT id, source_file_id FROM links WHERE target_path = ?", (path,)).fetchall()
            if links:
                print(f"  Found {len(links)} links with exact path: {path}")
                for link in links:
                    print(f"    ID {link['id']} from file {link['source_file_id']}")
            else:
                print(f"  No links with exact path: {path}")

if __name__ == '__main__':
    check_malformed_links()