#!/usr/bin/env python3

import os
import re
import sys

def load_progress_files():
    """Load existing files from progress.md and TO_DO.md"""
    progress_files = set()
    todo_files = set()

    # Read progress.md
    progress_file = os.path.join(os.path.dirname(__file__), '../Progress/progress.md')
    if os.path.exists(progress_file):
        try:
            with open(progress_file, 'r', encoding='utf-8') as f:
                lines = f.readlines()
            for line in lines[4:]:  # Skip header
                if line.strip().startswith('|'):
                    parts = [p.strip() for p in line.split('|')[1:]]
                    if len(parts) >= 3:
                        primary = parts[0]
                        folder = parts[1]
                        file_name = parts[2]
                        if primary and folder and file_name:
                            intranet_path = f"/intranet/{primary}/{folder}/{file_name}"
                            # Normalize the path
                            normalized_path = os.path.normpath(intranet_path)
                            progress_files.add(normalized_path)
        except Exception as e:
            print(f"Error reading progress.md: {str(e)}")

    # Read TO_DO.md
    todo_file = os.path.join(os.path.dirname(__file__), '../Progress/TO_DO.md')
    if os.path.exists(todo_file):
        try:
            with open(todo_file, 'r', encoding='utf-8') as f:
                lines = f.readlines()
            for line in lines:
                line = line.strip()
                if line and line.startswith('/intranet/'):
                    # Normalize the path to handle .. and .
                    normalized_path = os.path.normpath(line)
                    todo_files.add(normalized_path)
        except Exception as e:
            print(f"Error reading TO_DO.md: {str(e)}")

    return progress_files, todo_files

def convert_to_intranet_path(include_path, base_dir='../vmserver10/intranet'):
    """Convert include path to intranet format"""
    # Remove quotes and normalize path
    include_path = include_path.strip('\'"')

    # Handle absolute paths
    if include_path.startswith('/'):
        # Convert /databases/... to /intranet/databases/...
        if include_path.startswith('/databases/'):
            intranet_path = f"/intranet{include_path}"
        elif include_path.startswith('/intranet/'):
            intranet_path = include_path
        else:
            return None
    else:
        # Handle relative paths
        # Remove ../ or ./ prefixes
        include_path = re.sub(r'^(\.\./|\./)', '', include_path)
        intranet_path = f"/intranet/{include_path}"

    # Normalize the path
    return os.path.normpath(intranet_path)

def scan_file_for_php_references(file_path):
    """Scan PHP file for .php references"""
    if not os.path.exists(file_path):
        print(f"Error: File not found: {file_path}")
        return []

    try:
        with open(file_path, 'r', encoding='utf-8', errors='ignore') as f:
            content = f.read()
    except Exception as e:
        print(f"Error reading file: {str(e)}")
        return []

    php_files = set()

    # Find include/require statements with .php files
    include_patterns = [
        r'include\s*\(\s*[\'"]([^\'"]*\.php[^\'"]*)[\'"]\s*\)',
        r'include_once\s*\(\s*[\'"]([^\'"]*\.php[^\'"]*)[\'"]\s*\)',
        r'require\s*\(\s*[\'"]([^\'"]*\.php[^\'"]*)[\'"]\s*\)',
        r'require_once\s*\(\s*[\'"]([^\'"]*\.php[^\'"]*)[\'"]\s*\)',
    ]

    for pattern in include_patterns:
        matches = re.findall(pattern, content, re.IGNORECASE)
        for match in matches:
            intranet_path = convert_to_intranet_path(match)
            if intranet_path:
                php_files.add(intranet_path)

    return list(php_files)

def add_to_todo_list(new_files):
    """Add new files to TO_DO.md"""
    todo_file = os.path.join(os.path.dirname(__file__), '../Progress/TO_DO.md')

    try:
        # Read existing content and normalize paths
        existing_paths = set()
        if os.path.exists(todo_file):
            with open(todo_file, 'r', encoding='utf-8') as f:
                lines = f.readlines()
            for line in lines:
                line = line.strip()
                if line and line.startswith('/intranet/'):
                    normalized_path = os.path.normpath(line)
                    existing_paths.add(normalized_path)

        # Add new files
        with open(todo_file, 'a', encoding='utf-8') as f:
            for file_path in new_files:
                normalized_file_path = os.path.normpath(file_path)
                if normalized_file_path not in existing_paths:
                    f.write(f"{file_path}\n")
                    print(f"Added to TO_DO.md: {file_path}")
                    existing_paths.add(normalized_file_path)

    except Exception as e:
        print(f"Error writing to TO_DO.md: {str(e)}")

def main():
    if len(sys.argv) != 2:
        print("Usage: python3 Scanfile.py <file_path>")
        sys.exit(1)

    file_path = sys.argv[1]

    # Load existing files
    progress_files, todo_files = load_progress_files()

    # Scan for PHP references
    php_references = scan_file_for_php_references(file_path)

    # Check for new files
    new_files = []
    for ref in php_references:
        if ref not in progress_files and ref not in todo_files:
            new_files.append(ref)

    # Add to todo list
    if new_files:
        add_to_todo_list(new_files)
        print(f"SCANNED: {os.path.basename(file_path)} - Added {len(new_files)} new files")
    else:
        print(f"SCANNED: {os.path.basename(file_path)} - No new files found")

    # Return success
    sys.exit(0)

if __name__ == '__main__':
    main()